/*
 * Routines for dealing with fabric maps on the FMAs
 */
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/time.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_fma_comm.h"
#include "lf_topo_map.h"
#include "lf_channel.h"
#include "lf_scheduler.h"
#include "lf_fma_flags.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_notify.h"
#include "fms_fma.h"
#include "fms_fma_map.h"
#include "fms_state.h"

/*
 * local prototypes
 */
static void fms_assign_map_distribution(struct lf_fabric *fp);

/*
 * Create the topological representation of the fabric in a form suitable
 * for sending
 */
void
fms_create_topo_map(
  struct lf_fabric *fp)
{
  int topo_map_size;
  struct lf_topo_map *topo_map;
  int rc;

  /* free any previously existing topo map */
  LF_FREE(FMS_FABRIC(fp)->topo_map);

  /* If no hosts or xbars, then no need to proceed with a map */
  if (fp->num_hosts == 0 || fp->num_xbars == 0) {
    return;
  }

  /* perform some fabric processing */
  rc = lf_process_fabric_for_topo(fp);
  if (rc == -1) LF_ERROR(("Error processing fabric"));

  /* Assign map redistribution for nodes we cannot reach */
  fms_assign_map_distribution(fp);

  /* create the map */
  topo_map = lf_create_topo_map(fp, &topo_map_size);
  FMS_FABRIC(fp)->topo_map = topo_map;
  if (topo_map == NULL) {
    LF_ERROR(("Error creating topo_map"));
  }

  /*
   * Copy over and fill in some extras
   */
  FMS_FABRIC(fp)->topo_map_size = topo_map_size;
  fms_notify(FMS_EVENT_DEBUG, "topo_map_size = %d", topo_map_size);

  /* fill in mapper info */
  LF_MAC_COPY(topo_map->mapper_mac_addr, FMS_MAPPER_MAC);
  topo_map->mapper_level_16 = htons(FMA_FMS_MAPPER_LEVEL);
  ++F.fabvars->map_version;
  topo_map->map_id_32 = htonl(F.fabvars->map_version);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * update link state in the topo map
 */
void
fms_update_topo_link_state(
  union lf_node *np,
  int port)
{
  struct lf_fabric *fp;
  struct lf_topo_map *topo_map;
  lf_topo_link_state_t *lsp;
  int topo_index;

  fp = F.fabvars->fabric;
  topo_map = FMS_FABRIC(fp)->topo_map;

  if (topo_map != NULL) {
    topo_index = lf_get_node_link_topo_index(np, port);
    lsp = LF_TOPO_LINK_STATE_ARRAY(topo_map);
    lsp[topo_index] = lf_get_node_link_state(np, port);
  }
}

/*
 * Send the latest map to all registered FMAs that need it
 */
int
fms_send_topo_map_to_all_needy_fma(
  struct lf_fabric *fp)
{
  int h;

  /* There should be a map */
  if (FMS_FABRIC(fp)->topo_map == NULL) {
    LF_ERROR(("No map at push time!"));
  }

  fms_notify(FMS_EVENT_INFO, "Sending map to FMAs that need it");

  for (h=0; h<fp->num_hosts; ++h) {
    struct fms_fma_desc *adp;

    adp = FMS_HOST(fp->hosts[h])->adp;
    if (adp != NULL) {
      int rc;

      if (adp->needs_map) {
	fms_notify(FMS_EVENT_DEBUG, "Sending topo map to %s", 
	    fp->hosts[h]->hostname);
	rc = fms_send_topo_map(F.fabvars->fabric, adp);
	if (rc != 0) fms_perror();

      } else if (adp->needs_link_state) {
	rc = fms_send_topo_link_state(F.fabvars->fabric, adp);
	if (rc != 0) fms_perror();
      }
    }
  }

  /* update state machine that map has been sent */
  fms_state_map_sent();

  return 0;

 except:
  return -1;
}

/*
 * Mark every FMA as needing a map update
 */
void
fms_fma_invalidate_map()
{
  struct lf_fabric *fp;
  int h;

  fp = F.fabvars->fabric;

  fms_notify(FMS_EVENT_DEBUG, "All hosts need new map.");

  /* free the existing map */
  LF_FREE(FMS_FABRIC(fp)->topo_map);

  /* mark every host as needing a map */
  for (h=0; h<fp->num_hosts; ++h) {
    struct fms_fma_desc *adp;

    adp = FMS_HOST(fp->hosts[h])->adp;
    if (adp != NULL) {
      adp->needs_map = TRUE;
    }
  }

  return;
}

/*
 * Mark every FMA that doesn't need a map as needing a link state update
 */
void
fms_fma_invalidate_link_state(
  struct lf_fabric *fp)
{
  int h;

  fms_notify(FMS_EVENT_DEBUG, "All hosts need new link state.");

  /* mark every host as needing a map */
  for (h=0; h<fp->num_hosts; ++h) {
    struct fms_fma_desc *adp;

    adp = FMS_HOST(fp->hosts[h])->adp;
    if (adp != NULL && !adp->needs_map) {
      adp->needs_link_state = TRUE;
    }
  }
  return;
}

/*
 * Mark this FMA as needing a map, and schedule a map push
 */
void
fms_fma_needs_topo_map(
  struct fms_fma_desc *adp)
{
  struct lf_fabric *fp;

  fp = F.fabvars->fabric;

  adp->needs_map = TRUE;	/* mark this FMA as needing a map */

  /* Arrange for a map to be sent */
  fms_schedule_map_push();
}

/*
 * Is a map request pending?
 */
int
fms_map_request_pending()
{
  struct lf_fabric *fp;
  struct fms_fabric *fmsp;

  fp = F.fabvars->fabric;
  fmsp = FMS_FABRIC(fp);

  return (fmsp->request_map_timeout != NULL 
          || fmsp->request_map_task != NULL);
}

void
fms_schedule_map_request()
{
  struct lf_fabric *fp;
  struct fms_fabric *fmsp;

  fp = F.fabvars->fabric;
  fmsp = FMS_FABRIC(fp);

  /* If a map request is not pending and not already scheduled, schedule now */
  if (!fms_map_request_pending()) {
    fmsp->request_map_task = lf_schedule_event(fms_fma_request_map,
	                                        NULL, 5*1000);
  }
}

/*
 * Send the topological part of the map to an FMA
 */
int
fms_send_topo_map(
  struct lf_fabric *fp,
  struct fms_fma_desc *adp)
{
  int rc;

  /* send topo map message header */
  rc = fms_fma_write(adp, LF_FMA_TOPO_MAP,
                     FMS_FABRIC(fp)->topo_map, FMS_FABRIC(fp)->topo_map_size);
  if (rc == -1) {
    LF_ERROR(("Error sending topo_map header to %s", adp->hostname));
  }

  adp->needs_map = FALSE;	/* map no longer needed */
  adp->needs_link_state = FALSE;
  
  return 0;

 except:
  disconnect_from_fma(adp);
  return -1;
}

/*
 * Send the topological link state to an FMA
 */
int
fms_send_topo_link_state(
  struct lf_fabric *fp,
  struct fms_fma_desc *adp)
{
  int rc;
  int size;

  /* send topo link state message header */
  size = LF_TOPO_LINK_STATE_SIZE(FMS_FABRIC(fp)->topo_map);
  rc = fms_fma_write(adp, LF_FMA_TOPO_LINK_STATE,
       		LF_TOPO_LINK_STATE_ARRAY(FMS_FABRIC(fp)->topo_map), size);
  if (rc == -1) {
    LF_ERROR(("Error sending link state to %s", adp->hostname));
  }

  adp->needs_link_state = FALSE;	/* link state no longer needed */
  
  return 0;

 except:
  disconnect_from_fma(adp);
  return -1;
}

/*
 * For every node in the fabric we can't reach, assign a map re-distributor
 */
static void
fms_assign_map_distribution(
  struct lf_fabric *fp)
{
  struct lf_host *hp;
  int **src_table;
  int *src_cnt;
  int *src_next_index;
  int sf_id;
  int num_sf;
  int si;
  int i;
  int h;
  int t;

  /* Count number of subfabrics */
  t = 0;
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];
    t |= hp->subfabric_mask;
  }
  num_sf = 0;
  while (t > 0) {
    t >>= 1;
    ++num_sf;
  }

  /* if no subfabrics, all done! */
  if (num_sf == 0) return;

  /* allocate a table for each subfabric */
  LF_CALLOC(src_cnt, int, num_sf);
  LF_CALLOC(src_next_index, int, num_sf);
  LF_CALLOC(src_table, int *, num_sf);
  for (i=0; i<num_sf; ++i) {
    LF_CALLOC(src_table[i], int, fp->num_hosts);
  }

  /* Scan through all hosts in fabric, collecting and remembering those
   * capable of sending a map
   */
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    if (FMS_HOST(hp)->adp != NULL) {

      /* make sure no one sends to this guy */
      hp->distributor = fp->num_hosts;

      /* If this guy can distribute, add him to list */
      if (hp->fma_flags & FMA_FLAG_CAN_DISTRIBUTE) {
	for (i=0; i<num_sf; ++i) {

	  /* If we are in this subfabric, become a potential distributor */
	  if ((hp->subfabric_mask & (1<<i)) != 0) {
	    src_table[i][src_cnt[i]] = h;
	    ++src_cnt[i];
	  }
	}
      }
    }
  }

  /*
   * Scan through the list one more time, this time looking for
   * hosts that need redistribution and assigning them to the next distributor
   * in line.
   */
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    /* If we are in contact with this host, no distributor needed */
    if (FMS_HOST(hp)->adp != NULL) continue;

    /*
     * find a distributor for this node.  First we find the lowest subfabric
     * of this it is a member, then take the next distributor for that
     * subfabric.
     */
    t = hp->subfabric_mask;
    sf_id = 0;
    while (t > 1) {
      t >>= 1;
      ++sf_id;
    }

    /* take the top distributor from the list for this subfabric */
    si = src_next_index[sf_id];
    hp->distributor = src_table[sf_id][si];

    /* increment and wrap the next index */
    ++si;
    if (si >= src_cnt[sf_id]) {
      si = 0;
    }
    src_next_index[sf_id] = si;
  }

  /* free all of our allocated data */
  for (i=0; i<num_sf; ++i) {
    LF_FREE(src_table[i]);
  }
  LF_FREE(src_table);
  LF_FREE(src_cnt);
  LF_FREE(src_next_index);
  return;

 except:
  fms_perror_exit(1);
}
